<?php
header('Content-Type: application/json');
require 'connection.php';

function jsonResponse($status, $message, $extra = []) {
    echo json_encode(array_merge([
        'status' => $status,
        'message' => $message
    ], $extra));
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    jsonResponse('error', 'Only POST method allowed');
}

$rawInput = file_get_contents('php://input');
$input = json_decode($rawInput, true);

if (!$input) {
    http_response_code(400);
    jsonResponse('error', 'Invalid JSON input');
}

$client_id        = intval($input['client_id'] ?? 0);
$course_id        = intval($input['course_id'] ?? 0);
$year_id          = intval($input['year_id'] ?? 0);
$division_id      = intval($input['division_id'] ?? 0);
$faculty_id       = intval($input['faculty_id'] ?? 0);
$present_students = $input['present_students'] ?? [];

if (
    !$client_id || !$course_id || !$year_id ||
    !$division_id || !$faculty_id || !is_array($present_students)
) {
    http_response_code(400);
    jsonResponse('error', 'Missing required parameters');
}

$stmtSubject = $conn->prepare("
    SELECT id
    FROM uni_subjects
    WHERE client_id = ?
      AND course_id = ?
      AND year_id = ?
      AND division_id = ?
      AND deleted_at IS NULL
    LIMIT 1
");

$stmtSubject->bind_param("iiii", $client_id, $course_id, $year_id, $division_id);
$stmtSubject->execute();
$resSubject = $stmtSubject->get_result();

if ($resSubject->num_rows === 0) {
    jsonResponse('error', 'No subject found');
}

$subject_id = intval($resSubject->fetch_assoc()['id']);
$stmtSubject->close();

$stmtStudents = $conn->prepare("
    SELECT a.student_id
    FROM uni_student_academic a
    JOIN uni_stud_data s ON s.id = a.student_id
    WHERE a.client_id = ?
      AND s.course_id = ?
      AND a.year_id = ?
      AND a.division_id = ?
      AND a.status = 'active'
      AND s.status = 'active'
");

$stmtStudents->bind_param("iiii", $client_id, $course_id, $year_id, $division_id);
$stmtStudents->execute();
$resultStudents = $stmtStudents->get_result();

$students = [];
while ($row = $resultStudents->fetch_assoc()) {
    $students[] = intval($row['student_id']);
}
$stmtStudents->close();

if (empty($students)) {
    jsonResponse('error', 'No active students found');
}

$attendance_date = date('Y-m-d');
$scan_time = date('Y-m-d H:i:s');

$stmtCheck = $conn->prepare("
    SELECT id
    FROM uni_class_attend_log
    WHERE student_id = ?
      AND subject_id = ?
      AND attendance_date = ?
      AND client_id = ?
    LIMIT 1
");

$stmtUpdate = $conn->prepare("
    UPDATE uni_class_attend_log
    SET status = ?, scantime = ?, faculty_id = ?, updated_at = NOW()
    WHERE id = ?
");

$stmtInsert = $conn->prepare("
    INSERT INTO uni_class_attend_log
    (student_id, subject_id, attendance_date, status, scantime, taken_from, client_id, faculty_id, created_at)
    VALUES (?, ?, ?, ?, ?, 'faculty', ?, ?, NOW())
");

$present = 0;
$absent  = 0;

foreach ($students as $student_id) {

    $status = in_array($student_id, $present_students, true) ? 'P' : 'A';
    $status === 'P' ? $present++ : $absent++;
    
    $stmtCheck->bind_param("iisi", $student_id, $subject_id, $attendance_date, $client_id);
    $stmtCheck->execute();
    $resCheck = $stmtCheck->get_result();

    if ($resCheck->num_rows > 0) {
        
        $row = $resCheck->fetch_assoc();
        $log_id = intval($row['id']);

        $stmtUpdate->bind_param(
            "ssii",
            $status,
            $scan_time,
            $faculty_id,
            $log_id
        );
        $stmtUpdate->execute();

    } else {
        
        $stmtInsert->bind_param(
            "iisssii",
            $student_id,
            $subject_id,
            $attendance_date,
            $status,
            $scan_time,
            $client_id,
            $faculty_id
        );
        $stmtInsert->execute();
    }
}

$stmtCheck->close();
$stmtUpdate->close();
$stmtInsert->close();
$conn->close();

jsonResponse('success', 'Attendance saved', [
    'subject_id' => $subject_id,
    'present'    => $present,
    'absent'     => $absent,
    'total'      => count($students),
    'date'       => $attendance_date
]);
