<?php
header('Content-Type: application/json');
require 'connection.php';

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode([
        'status' => 'error',
        'message' => 'Only GET method allowed'
    ]);
    exit;
}

$client_id   = $_GET['client_id'] ?? '';
$course_id   = $_GET['course_id'] ?? '';
$year_id     = $_GET['year_id'] ?? '';
$division_id = $_GET['division_id'] ?? '';

if (!$client_id || !$course_id || !$year_id || !$division_id) {
    http_response_code(400);
    echo json_encode([
        'status' => 'error',
        'message' => 'Missing required parameters: client_id, course_id, year_id, division_id'
    ]);
    exit;
}

$names = [];

$stmt = $conn->prepare("SELECT name FROM uni_clients WHERE id = ? LIMIT 1");
$stmt->bind_param("i", $client_id);
$stmt->execute();
$result = $stmt->get_result()->fetch_assoc();
$names['institute_name'] = $result['name'] ?? 'Unknown Institute';
$stmt->close();

$stmt = $conn->prepare("SELECT name FROM uni_courses WHERE id = ? LIMIT 1");
$stmt->bind_param("i", $course_id);
$stmt->execute();
$result = $stmt->get_result()->fetch_assoc();
$names['course_name'] = $result['name'] ?? 'Unknown Course';
$stmt->close();

$stmt = $conn->prepare("SELECT name FROM uni_years WHERE id = ? LIMIT 1");
$stmt->bind_param("i", $year_id);
$stmt->execute();
$result = $stmt->get_result()->fetch_assoc();
$names['year_name'] = $result['name'] ?? 'Unknown Year';
$stmt->close();

$stmt = $conn->prepare("SELECT name FROM uni_divisions WHERE id = ? LIMIT 1");
$stmt->bind_param("i", $division_id);
$stmt->execute();
$result = $stmt->get_result()->fetch_assoc();
$names['division_name'] = $result['name'] ?? 'Unknown Division';
$stmt->close();

$students = [];

$stmt = $conn->prepare("
    SELECT 
        sd.id,
        sd.enrollment_no,
        sd.name,
        sa.roll_no
    FROM uni_stud_data sd
    INNER JOIN uni_student_academic sa ON sd.id = sa.student_id
    WHERE sa.client_id = ?
      AND sd.course_id = ?
      AND sa.year_id = ?
      AND sa.division_id = ?
      AND sd.status = 'active'
      AND sa.status = 'active'
    ORDER BY sa.roll_no
");
$stmt->bind_param("iiii", $client_id, $course_id, $year_id, $division_id);
$stmt->execute();
$res = $stmt->get_result();

while ($row = $res->fetch_assoc()) {
    $students[$row['id']] = [
        'enrollment_no' => $row['enrollment_no'],
        'roll_no'       => $row['roll_no'],
        'name'          => $row['name'],
        'present'       => 0,
        'total'         => 0,
        'percentage'    => 0
    ];
}
$stmt->close();

if (empty($students)) {
    echo json_encode([
        'status' => 'error',
        'message' => 'No students found for the specified division'
    ]);
    exit;
}

$months = [];
$totalAttendance = 0;
$totalPresent = 0;

$student_ids = array_keys($students);
$placeholders = implode(',', array_fill(0, count($student_ids), '?'));

$stmt = $conn->prepare("
    SELECT student_id, attendance_date, status, scantime
    FROM uni_class_attend_log
    WHERE student_id IN ($placeholders)
      AND client_id = ?
    ORDER BY attendance_date DESC, scantime DESC
");

$types = str_repeat('i', count($student_ids)) . 'i';
$params = array_merge($student_ids, [$client_id]);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$res = $stmt->get_result();

while ($row = $res->fetch_assoc()) {
    $student_id = $row['student_id'];
    $monthKey = date("F Y", strtotime($row['attendance_date']));
    $totalAttendance++;

    if (!isset($months[$monthKey])) {
        $months[$monthKey] = [
            'total' => 0,
            'present' => 0,
            'percentage' => 0,
            'history' => []
        ];
    }

    $months[$monthKey]['total']++;
    $students[$student_id]['total']++;

    if ($row['status'] === 'P') {
        $months[$monthKey]['present']++;
        $totalPresent++;
        $students[$student_id]['present']++;
    }

    $months[$monthKey]['history'][] = [
        'enrollment_no'   => $students[$student_id]['enrollment_no'],
        'roll_no'         => $students[$student_id]['roll_no'],
        'name'            => $students[$student_id]['name'],
        'attendance_date' => $row['attendance_date'],
        'scan_time'       => $row['scantime'],
        'status'          => $row['status']
    ];
}
$stmt->close();

foreach ($months as $k => $m) {
    $months[$k]['percentage'] =
        $m['total'] > 0
            ? round(($m['present'] / $m['total']) * 100, 2)
            : 0;
}

foreach ($students as $k => $s) {
    $students[$k]['percentage'] =
        $s['total'] > 0
            ? round(($s['present'] / $s['total']) * 100, 2)
            : 0;
}

$overall = $totalAttendance > 0
    ? round(($totalPresent / $totalAttendance) * 100, 2)
    : 0;

$conn->close();

echo json_encode([
    'status' => 'success',
    'institute' => $names['institute_name'],
    'course'    => $names['course_name'],
    'year'      => $names['year_name'],
    'division'  => $names['division_name'],
    'overall_percentage' => $overall,
    'students' => array_values($students),
    'months' => $months
]);