<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type");

require_once 'connection.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'status' => 'error',
        'message' => 'Only POST method allowed'
    ]);
    exit;
}

$login_id      = trim($_POST['login_id'] ?? '');
$client_id     = trim($_POST['client_id'] ?? '');
$security_code = trim($_POST['security_code'] ?? '');
$new_password  = trim($_POST['new_password'] ?? '');

if ($login_id === '' || $client_id === '' || $security_code === '' || $new_password === '') {
    http_response_code(400);
    echo json_encode([
        'status' => 'error',
        'message' => 'login_id, client_id, security_code and new_password are required'
    ]);
    exit;
}

$clientStmt = $conn->prepare("
    SELECT id
    FROM uni_clients
    WHERE id = ?
      AND status = 1
    LIMIT 1
");

$clientStmt->bind_param("i", $client_id);
$clientStmt->execute();

if ($clientStmt->get_result()->num_rows === 0) {
    http_response_code(404);
    echo json_encode([
        'status' => 'error',
        'message' => 'Invalid or inactive client'
    ]);
    exit;
}

$stmt = $conn->prepare("
    SELECT id, security_code
    FROM uni_faculties
    WHERE login_id = ?
      AND client_id = ?
      AND deleted_at IS NULL
    LIMIT 1
");

$stmt->bind_param("si", $login_id, $client_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    http_response_code(404);
    echo json_encode([
        'status' => 'error',
        'message' => 'Faculty not found for this client'
    ]);
    exit;
}

$user = $result->fetch_assoc();

if (!hash_equals($user['security_code'], $security_code)) {
    http_response_code(401);
    echo json_encode([
        'status' => 'error',
        'message' => 'Invalid security code'
    ]);
    exit;
}

$hashedPassword = password_hash($new_password, PASSWORD_DEFAULT);

$update = $conn->prepare("
    UPDATE uni_faculties
    SET password = ?, updated_at = NOW()
    WHERE id = ?
");

$update->bind_param("si", $hashedPassword, $user['id']);

if ($update->execute()) {
    echo json_encode([
        'status' => 'success',
        'message' => 'Password updated successfully'
    ]);
} else {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => 'Failed to update password'
    ]);
}

$conn->close();
